<?php

require_once __DIR__ . '/autoload.php';

use IAI\Authorization\OpenIdClient;
use IAI\Authorization\Oauth2Client;

if (PHP_VERSION_ID >= 70300) {
    session_set_cookie_params([
        'lifetime' =>  10800,
        'path' => '/; SameSite=None',
        'domain' => null,
        'secure' => true,
        'httponly' => true,
    ]);
} else {
    session_set_cookie_params(
        10800,
        '/; SameSite=None',
        null,
        true,
        true
    );
}
session_start();

// ---------- Application init ----------
$applicationState = new ApplicationState();
$keyStorage = new KeyStorage();


/**
 * ENTER YOUR APPLICATION'S CONFIGURATION BELOW
 * ENTER YOUR APPLICATION'S CONFIGURATION BELOW
 * ENTER YOUR APPLICATION'S CONFIGURATION BELOW
 */
$applicationConfig = (new \IAI\Application\Config())
    ->setPanelTechnicalDomain('panel.technical.domain')
    ->setId('client_id')
    ->setSecret('client_secret')
    ->setRedirectUri('https://yourappdomain/index.php');
//enter here exactly the same return address that you indicated on the page of individual application settings in the IdoSell Shop panel in the field "Address to which the application will be redirected after correct authorization"

$viewData = new stdClass();
// --------------------------------------


// ---------- Logging in, checking session, refreshing tokens etc. ----------
try {
    $viewData->errorMessage = '';

    if (!$applicationState->isLoggedIn()) {

        //Using OpenID Connect authentication server to log in
        $client = new OpenIdClient($applicationConfig, $keyStorage, $applicationState);

        if (empty($_GET['code']) && empty($_GET['state'])) {
            //no authorization code, no application state - first step of authentication - get the authorization code
            $client->startAuthentication();
        }

        //got (or should have) authorization code and application state - exchange authorization code for access token
        $token = $client->finalizeAuthentication($_GET['code'], $_GET['state']);

        //save received token in session
        $applicationState->setToken($token);

        //get logged in user details and save them in session
        $user = $token->getIdToken()->getClaim('profile', false);
        if ($user === false) {
            throw new Exception('Couldn\'t log in');
        }
        $applicationState->setUser($user);
    }

    if ($applicationState->hasToRefreshToken()) {
        //logged in, but have old authorization token - need to refresh token
        $client = new Oauth2Client($applicationConfig, $keyStorage);
        $applicationState->setToken(
            $client->refreshToken($applicationState->getToken())
        );
    }
} catch (Exception $e) {
    $viewData->errorMessage = $e->getMessage();
}

// --------------------------------------------------------------------------


// ---------- Getting orders data from API using OAuth authorization ----------
if (empty($viewData->errorMessage)) {
    $viewData->user = $applicationState->getUser();

    //creating SOAP client with Authorization header
    $gate = "https://{$applicationConfig->getPanelTechnicalDomain()}/api/?gate=orders/get/69/soap/wsdl";
    $apiClient = new SoapClient(
        $gate,
        [
            'stream_context' => stream_context_create([
                'http' => [
                    'header' => 'Authorization: Bearer ' . $applicationState->getToken()->getToken()
                ],
            ]),
            'cache_wsdl' => WSDL_CACHE_NONE
        ]
    );

    try {
        //building request
        $request = [
            'authenticate' => [
                //leaving empty - authenticating using OAuth access token
                'userLogin' => '',
                'authenticateKey' => ''
            ],
            'params' => [
                'ordersRange' => [
                    'ordersDateRange' => [
                        'ordersDateType' => 'add',
                        'ordersDateBegin' => ($date = date('Y-m-d')) . ' 00:00:00',
                        'ordersDateEnd' => $date . ' 23:59:59',
                    ]
                ]
            ]
        ];

        $page = 0;
        do {
            //fetching orders data
            $viewData->orders = [];
            $viewData->ordersWorthTotal = [];

            $request['params']['resultsPage'] = $page;
            $response = $apiClient->get($request);

            if ($response->errors && !empty($response->errors->faultString)) {
                $viewData->errorMessage = 'Error while executing API Orders: ' . $response->errors->faultString;
                break;
            }

            foreach ($response->Results as $order) {
                $orderWorthDetails = $order->orderDetails->payments->orderCurrency;
                $viewData->orders[$order->orderDetails->orderStatus]['count']++;
                $orderWorth =
                    $orderWorthDetails->orderProductsCost +
                    $orderWorthDetails->orderDeliveryCost +
                    $orderWorthDetails->orderPayformCost +
                    $orderWorthDetails->orderInsuranceCost;

                $viewData->orders[$order->orderDetails->orderStatus]['currencyTotal'][$orderWorthDetails->currencyId] += $orderWorth;
                $viewData->ordersWorthTotal[$orderWorthDetails->currencyId] += $orderWorth;
            }

            $page++;
        } while ($page < $response->resultsNumberPage);
    } catch (Exception $e) {
        $viewData->errorMessage = 'Error while executing API Orders: ' . $e->getMessage();
    }
}
// ----------------------------------------------------------------------------

// ------------------------------ Application "view" ------------------------------

/*
 * Getting CSS location in order to use IdoSell Shop panel CSS in application.
 * Don't worry about Same Origin Policy. Gate serves proper Access-Control-Allow-Origin header.
 */
$cssHref = ((new IAI\Application\Resources($applicationConfig))->getStyleSheetUrl());

?>

<html>

<head>
    <?php
    print '<link href="' . $cssHref . '" rel="stylesheet" type="text/css" />';
    ?>
</head>

<body>

    <?php
    if (!empty($viewData->errorMessage)) {
        print '
        <div class="msgWrapper " style="">
            <div class="alert alert-block alert-danger">
                <div style="display: table">
                    <div class="icon">
                        <i class="ace-icon fa  fa-exclamation-triangle bigger-200"></i>
                    </div>
                    <div class="message">
                        <strong>Attention:</strong> ' . $viewData->errorMessage . '
                    </div>
                </div>
            </div>
        </div>';
    }

    if (!empty($viewData->user)) {
        print "
        <h2 style=\"text-align: left;\">Logged as: {$viewData->user->name} ({$viewData->user->preferred_username})</h2>
        <br /><br />
    ";

        if (!empty($viewData->orders)) {
            //displaying fetched orders data

            print '
            <table class="t4 table table-striped table-bordered table-hover" cellpadding="2" cellspacing="2" align="center">
                <tbody>
                    <tr>
                        <td class="title">Order status</td><td class="title">' .
                implode('</td><td class="title">', array_keys($viewData->orders)) . '</td>
                    </tr>';

            $currencyFormatter = function ($currencyId, $value) {
                return sprintf('%.2f', $value) . ' ' . $currencyId;
            };

            $rows = [
                0 => ['Count'],
                1 => ['Value by currency'],
            ];

            $columnsCount = 1;

            foreach ($viewData->orders as $orderStatus => $ordersInfo) {
                $rows[0][] = $ordersInfo['count'];
                $totalByCurrency = [];
                foreach ($ordersInfo['currencyTotal'] as $currencyId => $ordersTotal) {
                    $totalByCurrency[] = $currencyFormatter($currencyId, $ordersTotal);
                }
                $rows[1][] = implode('<br />', $totalByCurrency);
                $columnsCount++;
            }

            foreach ($rows as $row) {
                print '<tr class="row2"><td class="row0">' . implode('</td><td class="row0">', $row) . '</td></tr>';
            }
            foreach ($viewData->ordersWorthTotal as $currencyId => $ordersTotal) {
                $viewData->ordersWorthTotal[$currencyId] = $currencyFormatter($currencyId, $ordersTotal);
            }

            print '<tr class="row2"><td class="row0" colspan="' . $columnsCount . '"/></tr>
               <tr class="row2">
                    <td class="row0" colspan="' . ($columnsCount - 1) . '" style="text-align: right;">Altogether</td>
                    <td class="row0">' . implode('<br />', $viewData->ordersWorthTotal) . '</td>
               </tr>';

            print '
                </tbody>
            </table>';
        } else {
            print 'No orders today (yet).';
        }
    }


    ?>
</body>

</html>

<?php
// --------------------------------------------------------------------------------