<?php

namespace IAI\Authorization;

use IAI\Application;
use Lcobucci\JWT;

/**
 * Abstract of authentication / authorization provider
 */
abstract class AbstractClient
{
    /**
     * Application config
     *
     * @var Application\Config
     */
    protected $config;

    /**
     * Public key storage for OpenID Connect / OAuth2 public encryption key
     *
     * @var Application\PublicKeyStorageInterface
     */
    protected $keyStorage;

    /**
     * OpenID Connect / OAuth2 provider client
     *
     * @var \League\OAuth2\Client\Provider\AbstractProvider
     */
    protected $provider;

    /**
     * Public encryption key
     *
     * @var string
     */
    protected $publicKey;

    /**
     * Signer library
     *
     * @var JWT\Signer\Rsa\Sha256
     */
    protected $signer;

    /**
     * AbstractClient constructor.
     *
     * @param \IAI\Application\Config                    $config Application config
     * @param \IAI\Application\PublicKeyStorageInterface $keyStorage Key storage for OpenID Connect / OAuth2 public encryption key
     *
     * @throws \IAI\Authorization\Exception\InvalidPublicKeyException
     */
    public function __construct(Application\Config $config, Application\PublicKeyStorageInterface $keyStorage)
    {
        $this->config = $config;
        $this->keyStorage = $keyStorage;
        $this->signer = new JWT\Signer\Rsa\Sha256();
        $this->provider = $this->initProvider();
    }

    /**
     * Initializes and returns OpenID Connect / OAuth2 provider client
     *
     * @return \League\OAuth2\Client\Provider\AbstractProvider
     *
     * @throws \IAI\Authorization\Exception\InvalidPublicKeyException
     */
    abstract protected function initProvider();

    /**
     * Retrieves public key
     *
     * @return string
     *
     * @throws \IAI\Authorization\Exception\InvalidPublicKeyException
     */
    protected function getPublicKey()
    {
        $key = $this->keyStorage->retrieve();
        if (empty($key)) {
            $key = file_get_contents($this->getPublicKeyUrl());
            if (empty($key)) {
                throw new Exception\InvalidPublicKeyException(
                    'Key obtained from issuer ' . $this->config->getPanelTechnicalDomain() . ' is empty'
                );
            }
            $this->keyStorage->store($key);

            //storage must work properly :)
            $key = $this->keyStorage->retrieve();
            if (empty($key)) {
                throw new Exception\InvalidPublicKeyException('Couldn\'t retrieve public key from storage');
            }
        }

        return $key;
    }

    /**
     * Returns provider OpenID Connect / OAuth2 endpoint url
     *
     * @return string
     */
    private function getProviderEndpointUrl()
    {
        return 'https://' . $this->config->getPanelTechnicalDomain() . '/panel/action/authorize/';
    }

    /**
     * Returns issuer public key location
     *
     * @return string
     */
    protected function getPublicKeyUrl()
    {
        return $this->getProviderEndpointUrl() . 'public_key';
    }

    /**
     * Returns authorization endpoint location
     *
     * @return string
     */
    protected function getAuthorizeUrl()
    {
        return $this->getProviderEndpointUrl() . 'authorize';
    }

    /**
     * Returns access token endpoint location
     *
     * @return string
     */
    protected function getAccessTokenUrl()
    {
        return $this->getProviderEndpointUrl() . 'access_token';
    }
}